<?php
/**
 * Public Class
 *
 * @package     Wow_Plugin
 * @subpackage  Public
 * @author      Wow-Company <yoda@wow-company.com>
 * @copyright   2019 Wow-Company
 * @license     GNU Public License
 * @version     1.0
 */

namespace side_menu_pro;

defined( 'ABSPATH' ) || exit;

/**
 * Class Wow_Plugin_Public
 *
 * @package wow_plugin
 *
 * @property array plugin   - base information about the plugin
 * @property array url      - home, pro and other URL for plugin
 * @property array rating   - website and link for rating
 * @property string basedir  - filesystem directory path for the plugin
 * @property string baseurl  - URL directory path for the plugin
 */
class Wow_Plugin_Public {

	/**
	 * Setup to frontend of the plugin
	 *
	 * @param array $info general information about the plugin
	 *
	 * @since 1.0
	 */

	public function __construct( $info ) {
		$this->plugin = $info['plugin'];
		$this->url    = $info['url'];
		$this->rating = $info['rating'];

		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
		add_shortcode( $this->plugin['shortcode'], [ $this, 'shortcode' ] );

		add_action( $this->plugin['slug'] . '_include_style', [ $this, 'include_style' ] );
		add_action( $this->plugin['slug'] . '_include_fontawesome', [ $this, 'include_fontawesome_css' ] );

		$this->includes();
		$this->conditions = new Conditions( $this->plugin );
		$this->display    = new Display_Item( $this->plugin );
	}

	public function include_style( $param = '' ) {
		if ( empty( $param ) ) {
			return false;
		}
		$style_file = is_rtl() ? 'style-rtl' : 'style';
		$style_path = plugin_dir_url( __FILE__ ) . 'assets/css/' . $style_file . '.css';

		wp_enqueue_style( 'side-menu', $style_path, null, $this->plugin['version'] );

		if ( ! empty( $param['style'] ) ) {
			$data = trim( preg_replace( '~\s+~s', ' ', $param['style'] ) );
			wp_add_inline_style( 'side-menu', $data );
		}
	}

	public function include_fontawesome_css( $param = '' ) {
		if ( empty( $param['disable_fontawesome'] ) ) {
			$fontawesome_path = $this->plugin['url'] . 'vendors/fontawesome/css/fontawesome-all.min.css';
			wp_enqueue_style( 'fontawesome', $fontawesome_path, null, '6.1.2' );
		}
	}

	public function enqueue_scripts() {
		if ( is_singular() ) {
			global $post;
			$post_content = $post->post_content;
			$param        = '';

			if ( ! empty( $post_content ) && has_shortcode( $post_content, $this->plugin['shortcode'] ) ) {
				$id = $this->get_shortcode_id( $post_content );

				if ( ! empty( $id ) ) {
					$param = $this->get_parameters( $id );

					$this->include_fontawesome_css( $param );
				}
			}
			$this->include_style( $param );
		}
	}

	public function get_shortcode_id( $content ) {
		$pattern = get_shortcode_regex();
		$id      = 0;

		if ( preg_match_all( '/' . $pattern . '/s', $content, $matches ) ) {
			$shortcodeIndex = array_search( $this->plugin['shortcode'], $matches[2] );
			if ( $shortcodeIndex !== false ) {
				$att       = $matches[3][ $shortcodeIndex ];
				$id_string = explode( "=", $att )[1];
				$id        = str_replace( [ '"', '\'' ], '', $id_string );
			}
		}

		return absint( $id );
	}

	public function get_parameters( $id ) {
		global $wpdb;
		$table  = $wpdb->prefix . 'wow_' . $this->plugin['prefix'];
		$result = $wpdb->get_var( $wpdb->prepare( "SELECT param FROM $table WHERE id = %d", $id ) );

		return empty( $result ) ? false : unserialize( $result );
	}

	public function includes() {
		require_once 'class-display.php';
		require_once 'class-conditions.php';
		require_once 'class-walker.php';
		require_once 'class-js-object-generator.php';
	}


	public function conditions( $param ) {
		return $this->conditions->check( $param );
	}

	/**
	 * Display a shortcode
	 *
	 * @param $atts
	 *
	 * @return false|string
	 */
	public function shortcode( $atts ) {
		extract( shortcode_atts( [ 'id' => '' ], $atts ) );
		$id    = absint( $id );
		$param = $this->get_parameters( $id );

		if ( empty( $param ) || ! $this->conditions( $param ) || empty( $param['menu_1']['item_type'])) {
			return false;
		}

		$walker = new Maker_Menu();
		$menu   = $walker->display_menu( $id, $param );

		$this->include_style_script( $param, $id );

		return $menu;
	}

	private function include_style_script( $param, $id ) {
		$slug    = $this->plugin['slug'];
		$version = $this->plugin['version'];

		do_action( $this->plugin['slug'] . '_include_style', $param );
		do_action( $this->plugin['slug'] . '_include_fontawesome', $param );

		$url_script = plugin_dir_url( __FILE__ ) . 'assets/js/script.js';
		wp_enqueue_script( $slug, $url_script, [], $version );

		$inline_script = $this->script( $param, $id );
		wp_add_inline_script( $slug, $inline_script, 'before' );
	}


	/**
	 * Create Inline script for elements
	 */
	public function script( $param, $id ) {
		return ( new JS_Object_Generator )->generate( $id, $param );
	}


}
