<?php
/**
 * Class to handle compatibilities with different themes and plugins.
 */
class BM_Compatibilities {

	/**
	 * Contains instance or null
	 *
	 * @var object|null
	 */
	private static $instance = null;

	/**
	 * Returns instance of BM_Price.
	 *
	 * @return object
	 */
	public static function get_instance() {

		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor for BM_Price.
	 */
	public function __construct() {
		$theme = wp_get_theme();

		// Prices.
		$prices = BM_Price::get_instance();

		/**
		 * German Market / Base Unit Prices
		 */
		add_filter('german_market_ppu_co_woocommerce_cart_item_price_ppu_string', function( $ppu_string, $cart_item, $cart_item_key ) {
			$price_per_unit = 0;

			// Find the correct product by ID to use.
			$product = wc_get_product( $cart_item['product_id'] );

			if ( isset( $cart_item['variation_id'] ) && ! empty( $cart_item['variation_id'] ) ) {
				$product = wc_get_product( $cart_item['variation_id'] );
			}

			$product_price = $product->get_regular_price();

			if ( $product->get_sale_price() > 0 ) {
				$product_price = $product->get_sale_price();
			}

			// get pricing data.
			$cheapest_price = BM_Price::get_price( $product_price, $product, false, $cart_item['quantity'] );
			$cheapest_price = BM_Tax::get_tax_price( $product, $cheapest_price );

			$cheapest_price = apply_filters( 'bm_filter_bundle_cheapest_price', $cheapest_price, $product, $cart_item );

			if ( $product->is_type( 'variation' ) ) {
				$variation_mult = get_post_meta( $product->get_id(), '_v_unit_regular_price_per_unit_mult', true );
				$variation_qty  = get_post_meta( $product->get_id(), '_v_auto_ppu_complete_product_quantity', true );
				$var_ppu_set    = get_post_meta( $product->get_id(), '_v_used_setting_ppu', true );

				if ( 1 == $var_ppu_set ) {
					$price_per_unit = WGM_Price_Per_Unit::automatic_calculation( $cheapest_price, $variation_qty, $variation_mult );
				} else {
					$parent_product = wc_get_product( $product->get_parent_id() );
					$ppu            = WGM_Price_Per_Unit::get_price_per_unit_data( $parent_product );

					if ( isset( $ppu['mult'] ) ) {
						$qty            = get_post_meta( $product->get_parent_id(), '_auto_ppu_complete_product_quantity', true );
						$price_per_unit = WGM_Price_Per_Unit::automatic_calculation( $cheapest_price, $qty, $ppu['mult'] );
					}
				}
			} else {
				$ppu = WGM_Price_Per_Unit::get_price_per_unit_data( $product );

				if ( isset( $ppu['mult'] ) ) {
					$qty            = get_post_meta( $product->get_id(), '_auto_ppu_complete_product_quantity', true );
					$price_per_unit = WGM_Price_Per_Unit::automatic_calculation( $cheapest_price, $qty, $ppu['mult'] );
				}
			}
			if ( $price_per_unit > 0 ) {
				$result = apply_filters( 'wmg_price_per_unit_loop',
					sprintf( '<span class="wgm-info price-per-unit price-per-unit-loop ppu-variation-wrap">' . trim( WGM_Price_Per_Unit::get_prefix( $ppu ) . ' ' . WGM_Price_Per_Unit::get_output_format() ) . '</span>',
						wc_price( $price_per_unit ),
						str_replace( '.', wc_get_price_decimal_separator(), $ppu[ 'mult' ] ),
						$ppu[ 'unit' ]
					),
					wc_price( str_replace( ',', '.', $ppu[ 'price_per_unit' ] ) ),
					$ppu[ 'mult' ],
					$ppu[ 'unit' ]
				);

				return $result;
			}
		}, 10, 3 );

		/**
		 * Germanized / Base Unit Prices
		 */
		if ( class_exists( 'WooCommerce_Germanized' ) ) {
			$group_id = BM_Conditionals::get_validated_customer_group();
			if ( ! empty( $group_id ) ) {
				$group      = get_post( $group_id );
				$group_slug = strtolower( $group->post_name );
				$hide_price = get_option( 'bm_hide_price_' . $group_slug, 'off' );
				if ( 'on' === $hide_price ) {
					add_filter( 'woocommerce_gzd_formatted_unit_price', function( $html, $price, $unit_base, $unit ) {
						return '';
					}, 10, 4 );
				}
				if ( class_exists( 'WooCommerce_Germanized_Pro' ) ) {

					add_filter( 'woocommerce_get_price_html', function( $html, $product ) {

						add_action( 'woocommerce_gzd_before_get_unit_price', function( $gzd_product ) {
							$gzd_product->recalculate_unit_price();
						}, 10, 1 );
						// Adjust variable from-to unit prices
						add_action( 'woocommerce_gzd_before_get_variable_variation_unit_price', function( $gzd_product ) {
							$gzd_product->recalculate_unit_price();
						}, 10, 1 );

						return $html;
					}, 200, 2 );

				}
			}
		}

		/**
		 * WP Bakery Pagebuilder.
		 */
		add_action( 'woocommerce_shortcode_before_featured_products_loop', array( $prices, 'reenable_prices' ) );
		add_action( 'woocommerce_shortcode_before_product_category_loop', array( $prices, 'reenable_prices' ) );

		/**
		 * Impreza theme.
		 */
		if ( 'Impreza' === $theme->name || 'Impreza' == $theme->parent_theme ) {
			add_action( 'us_before_template:templates/content', array( $prices, 'reenable_prices' ) );
		}

		/**
		 * Avada.
		 */
		if ( 'Avada' === $theme->name || 'Avada' == $theme->parent_theme ) {
			add_action( 'avada_after_header_wrapper', function() {
				add_filter( 'bm_filter_price', '__return_false' );
			});

			add_action( 'avada_before_main', array( $prices, 'reenable_prices' ) );
		}

		/**
		 * Divi Theme.
		 */
		if ( 'Divi' === $theme->name || 'Divi' == $theme->parent_theme ) {
			add_action( 'et_before_main_content', array( $prices, 'reenable_prices' ) );
		}

		/**
		 * Envo eCommerce Theme.
		 */
		if ( 'Envo eCommerce' === $theme->name || 'Envo eCommerce' == $theme->parent_theme ) {
			add_filter( 'bm_filter_price', '__return_false' );
		}

		/**
		 * Thrive Theme Builder
		 */
		if ( 'Thrive Theme Builder' === $theme->name || 'Thrive Theme Builder' == $theme->parent_theme ) {
			add_filter( 'bm_filter_price', '__return_false' );
		}

		/**
		 * Oxygen Elements for WooCommerce Plugin
		 */
		add_action( 'after_setup_theme', function() {
			if ( class_exists( 'OxyWooCommerce' ) ) {
				add_filter( 'bm_filter_price', '__return_false' );
			}
		});

		/**
		 * WooCommerce Chained Products Plugin
		 */
		add_action( 'after_setup_theme', function() {
			if ( class_exists( 'WC_Chained_Products' ) ) {
				$prices = BM_Price::get_instance();
				add_filter( 'bm_filter_price', '__return_false' );
			}
		});

		/**
		 * Elementor Plugin
		 */
		add_action( 'elementor/element/before_section_start', array( $prices, 'reenable_prices' ) );

		/**
		 * WooCommerce Product Table.
		 */
		if ( is_plugin_active( 'woocommerce-product-table/woocommerce-product-table.php' ) ) {
			add_filter( 'wc_product_table_data_price', function( $price, $product ) {
				$group_id       = BM_Conditionals::get_validated_customer_group();
				$cheapest_price = BM_Price::get_price( $product->get_price(), $product, $group_id, apply_filters( 'bm_default_qty', 1 ) );

				// RRP.
				$use_rrp   = get_option( 'bm_use_rrp', 'off' );
				$rrp_label = get_option( 'bm_rrp_label', __( 'RRP', 'b2b-market' ) );
				$rrp_price = floatval( get_post_meta( $product->get_id(), 'bm_rrp', true ) );

				if ( ! $product->is_type( 'variable' ) ) {
					if ( 'on' === $use_rrp ) {
						return '<small class="b2b-rrp">' . $rrp_label . ': ' . wc_price( $rrp_price ) . '</small><br>' . wc_price( $cheapest_price );
					} else {
						return wc_price( $cheapest_price );
					}
				}
				return $price;
			}, 10, 2 );

			// Temporary fix for variation prices.
			add_action( 'wp_footer', function() {
				?>
				<style>
				.wpt_variations_form .woocommerce-variation-price .price {
					display: none !important;
				}
				</style>
				<?php
			});
		}

		/**
		 * Flatsome
		 */
		if ( 'Flatsome' === $theme->name || 'Flatsome' == $theme->parent_theme ) {
			add_action( 'flatsome_after_header_bottom', array( $prices, 'reenable_prices' ) );
		}

		/**
		 * WooCommerce Product Bundles
		 */
		add_action( 'init', function() {
			if ( class_exists( 'WC_Bundles' ) ) {
				add_filter( 'bm_filter_get_sale_price_context', function( $context ) {
					$context = 'edit';
					return $context;
				} );
				add_filter( 'woocommerce_bundled_item_price', function( $price, $product, $discount, $bundled_item ) {

					if ( is_product() ) {
						add_filter( 'bm_filter_disable_bulk_price_table_show_totals', '__return_true' );
					}

					$product          = wc_get_product( $product->get_id() );
					$offset_price     = ! empty( $product->bundled_price_offset ) ? $product->bundled_price_offset : false;
					$offset_price_pct = ! empty( $product->bundled_price_offset_pct ) && is_array( $product->bundled_price_offset_pct ) ? $product->bundled_price_offset_pct : false;

					$group_id         = BM_Conditionals::get_validated_customer_group();
					$cheapest_price   = BM_Price::get_price( $product->get_price( 'edit' ), $product, $group_id, apply_filters( 'bm_default_qty', 1 ) );

					$cheapest_price   = WC_PB_Product_Prices::get_discounted_price($cheapest_price, $discount );

					// Add-on % prices.
					if ( $offset_price_pct ) {

						if ( ! $offset_price ) {
							$offset_price = 0.0;
						}

						foreach ( $offset_price_pct as $price_pct ) {
							$offset_price += $cheapest_price * $price_pct / 100;
						}
					}

					// Add-on prices.
					if ( $offset_price ) {
						$cheapest_price += $offset_price;
					}

					$product->bundled_item_price = $cheapest_price;

					return $cheapest_price;
				}, 10, 4 );
				add_filter( 'bm_filter_bundle_cheapest_price', function( $cheapest_price, $product, $cart_item ) {

					$offset_price     = ! empty( $product->bundled_price_offset ) ? $product->bundled_price_offset : false;
					$offset_price_pct = ! empty( $product->bundled_price_offset_pct ) && is_array( $product->bundled_price_offset_pct ) ? $product->bundled_price_offset_pct : false;

					if ( array_key_exists( 'bundle_sell_discount', $cart_item ) ) {
						$discount = intval( $cart_item[ 'bundle_sell_discount' ] );
					} else {
						$discount = 0;
					}

					$cheapest_price   = WC_PB_Product_Prices::get_discounted_price($cheapest_price, $discount );

					// Add-on % prices.
					if ( $offset_price_pct ) {

						if ( ! $offset_price ) {
							$offset_price = 0.0;
						}

						foreach ( $offset_price_pct as $price_pct ) {
							$offset_price += $cheapest_price * $price_pct / 100;
						}
					}

					// Add-on prices.
					if ( $offset_price ) {
						$cheapest_price += $offset_price;
					}

					$product->bundled_item_price = $cheapest_price;

					return $cheapest_price;
				}, 10, 3 );
			}
		} );

		/** WooCommerce Subscriptions */
		add_action( 'init', function() {
			if ( class_exists( 'WC_Subscriptions' ) ) {
				// Sale Price.
				add_filter( 'bm_filter_get_sale_price_context', function( $context ) {
					return 'edit';
				}, 10, 1 );

				// Regular Price.
				add_filter( 'bm_filter_get_price_context', function( $context ) {
					return 'edit';
				}, 10, 1 );
			}
		} );

		/**
		 * WPC Composite Products Plugin
		 */
		add_action( 'after_setup_theme', function() {
			if ( class_exists( 'WC_Product_Composite' ) ) {
				add_filter( 'wooco_product_original_price', function( $price, $product ) {
					$group_id       = BM_Conditionals::get_validated_customer_group();
					$cheapest_price = BM_Price::get_price( $product->get_price(), $product, $group_id, apply_filters( 'bm_default_qty', 1 ) );
					return $cheapest_price;
				}, 10, 2 );
				add_filter( 'bm_filter_get_price_context', function( $context ) {
					$context = 'edit';
					return $context;
				} );
			}
		} );

		/**
		 * MarketPress Salesman - Addon 'Free Shipping Notice'
		 */
		add_action( 'after_setup_theme', function() {
			if ( class_exists( 'CSP_ShippingManager' ) && class_exists( 'MarketPress_Salesman' ) && class_exists( 'MarketPress_Salesman_Free_Shipping_Notice' ) ) {
				$group_id = BM_Conditionals::get_validated_customer_group();
				if ( ! empty( $group_id ) ) {
					$group      = get_post( $group_id );
					$group_slug = $group->post_name;
					$status     = get_option( 'bm_shipping_method_enable_free_shipping_' . $group_slug );
					if ( 'off' === $status ) {
						$fsn      = MarketPress_Salesman_Free_Shipping_Notice::get_instance();
						$instance = $fsn->application;
						remove_action( 'woocommerce_before_cart_table', array( $instance, 'free_shipping_cart_notice' ) );
						add_filter( 'marketpress_salesman_free_shipping_notice_shortcode', function( $content ) {
							$content = '';
							return $content;
						} );
					}
				}
			}
		} );

		/**
		 * MarketPress Salesman - Addon 'Printy Coupons'
		 */
		add_action( 'after_setup_theme', function() {
			add_filter( 'bm_filter_get_regular_price', function( $price, $item ) {
				if ( class_exists( 'MarketPress_Salesman_Printy_Coupons' ) && 'voucher' == $item[ 'data' ]->get_type() ) {
					if ( is_cart() && ! empty( $item[ 'voucher_value' ] ) ) {
						$price = $item[ 'voucher_value' ];
					} else {
						$price = $item[ 'data' ]->get_price();
					}
				}
				return $price;
			}, 10, 2 );
			if ( class_exists( 'MarketPress_Salesman_Printy_Coupons' ) ) {
				add_filter( 'marketpress_salesman_printy_coupons_voucher_min_value', function ( $price, $product ) {
					$cheapest_price = BM_Price::get_price( $price, $product, false, 1 );
					if ( $cheapest_price < $price ) {
						$price = $cheapest_price;
					}
					return $price;
				}, 10, 2 );
				add_filter( 'marketpress_salesman_printy_coupons_voucher_max_value', function ( $price, $product ) {
					$cheapest_price = BM_Price::get_price( $price, $product, false, 1 );
					if ( $cheapest_price < $price ) {
						$price = $cheapest_price;
					}
					return $price;
				}, 10, 2 );
			}
		} );

	}
}
